
from random import seed
from random import randrange
from csv import reader
from sklearn.datasets import load_breast_cancer
from sklearn.datasets import load_iris
from sklearn.datasets import load_digits
from sklearn.datasets import load_wine
from math import pow
import scipy.stats
import matplotlib.pyplot as plt
import numpy as np
from milksets import seeds
from milksets import yeast

# Load CSV file
def load_csv(filename):
	file = open(filename, "rt")
	lines = reader(file)
	dataset = list(lines)
	return dataset

# Convert str column to float
def str_column_to_float(dataset, column):
	for row in dataset:
		row[column] = float(row[column].strip())

# Split dataset into k folds
def cross_validation_split(dataset, n_folds):
	dataset_split = list()
	dataset_copy = list(dataset)
	fold_size = int(len(dataset) / n_folds)
	for i in range(n_folds):
		fold = list()
		while len(fold) < fold_size:
			index = randrange(len(dataset_copy))
			fold.append(dataset_copy.pop(index))
		dataset_split.append(fold)
	return dataset_split

# Calculate acc percentage
def accuracy_metric(actual, predicted):
	correct = 0
	for i in range(len(actual)):
		if actual[i] == predicted[i]:
			correct += 1
	return correct / float(len(actual)) * 100.0

# Evaluate algorithm using a cross validation split
def evaluate_algorithm(dataset, algorithm, n_folds, *args):
	folds = cross_validation_split(dataset, n_folds)
	scores = list()
	sizes = list()
	for fold in folds:
		train_set = list(folds)
		train_set.remove(fold)
		train_set = sum(train_set, [])
		test_set = list()
		for row in fold:
			row_copy = list(row)
			test_set.append(row_copy)
			row_copy[-1] = None
		predicted, size = algorithm(train_set, test_set, *args)
		actual = [row[-1] for row in fold]
		accuracy = accuracy_metric(actual, predicted)
		scores.append(accuracy)
		sizes.append(size)
	return scores, sizes

# Split dataset based on an attribute and an attribute value
def test_split(index, value, dataset):
	left, right = list(), list()
	for row in dataset:
		if row[index] < value:
			left.append(row)
		else:
			right.append(row)
	return left, right

# Calculate the a,b-Tsallis entropy for a split dataset
def ab_tsallis(groups, classes, a, b):
	# count all samples at split point
	n_instances = float(sum([len(group) for group in groups]))
	# sum weighted Gini index for each group
	abt = 0.0
	for group in groups:
		size = float(len(group))
		# avoid division by zero
		if size == 0:
			continue
		score = 0.0
		# score group based on the score for each class
		for class_val in classes:
			p = [row[-1] for row in group].count(class_val) / size
			score += pow(p, a)
		tsallis = 1.0 - pow(score, b) if a > 1 else pow(score, b) - 1.0
		# weight group score by its relative size
		abt += tsallis * (size / n_instances)
	return abt

# Select best split point for a dataset
def get_split(dataset, a, b):
	class_values = list(set(row[-1] for row in dataset))
	b_index, b_value, b_score, b_groups = 999, 999, 999, None
	for index in range(len(dataset[0])-1):
		for row in dataset:
			groups = test_split(index, row[index], dataset)
			split_score = ab_tsallis(groups, class_values, a, b)
			if split_score < b_score:
				b_index, b_value, b_score, b_groups = index, row[index], split_score, groups
	return {'index':b_index, 'value':b_value, 'groups':b_groups}

# Create terminal node value
def to_terminal(group):
	outcomes = [row[-1] for row in group]
	return max(set(outcomes), key=outcomes.count)

def err_group(group):
    if len(group)==0:
        return 0
    label = to_terminal(group)
    err = [0 if row[-1]==label else 1 for row in group]
    return sum(err)/len(group)

# Create child splits for a node or make terminal
def split(node, max_depth, min_size, depth, a, b):
	left, right = node['groups']
	node['eleft'] = err_group(left)
	node['eright'] = err_group(right)
	node['nleft'] = len(left)
	node['nright'] = len(right)
	node['gleft'] =   to_terminal(left) if len(left)>0 else -1
	node['gright'] = to_terminal(right) if len(right)>0 else -1
	del(node['groups'])
	# check for a no split
	if not left or not right:
		node['left'] = node['right'] = to_terminal(left + right)
		return
	# check for max depth
	if depth >= max_depth:
		node['left'], node['right'] = to_terminal(left), to_terminal(right)
		return
	# process left child
	if len(left) <= min_size:
		node['left'] = to_terminal(left)
	else:
		node['left'] = get_split(left, a, b)
		split(node['left'], max_depth, min_size, depth+1, a, b)
	# process right child
	if len(right) <= min_size:
		node['right'] = to_terminal(right)
	else:
		node['right'] = get_split(right, a, b)
		split(node['right'], max_depth, min_size, depth+1, a, b)

# Build decision tree
def build_tree(train, max_depth, min_size, a, b):
	root = get_split(train, a, b)
	split(root, max_depth, min_size, 1, a, b)
	return root

# Make prediction with a decision tree
def predict(node, row):
	if row[node['index']] < node['value']:
		if isinstance(node['left'], dict):
			return predict(node['left'], row)
		else:
			return node['left']
	else:
		if isinstance(node['right'], dict):
			return predict(node['right'], row)
		else:
			return node['right']


# returns total error of leaf nodes, and total number of leaf nodes in subtree
def traverse_subtree(node):
    err = 0
    nl = 0
    if isinstance(node['left'], dict):
        e, l = traverse_subtree(node['left'])
        err += e
        nl += l
    else:
        err += node['eleft']*node['nleft']
        nl += 1
    if isinstance(node['right'], dict):
        e, l = traverse_subtree(node['right'])
        err += e
        nl += l
    else:
        err += node['eright']*node['nright']
        nl += 1
    return err, nl
    

def prune_tree(node, alpha):
    e_node = node['eleft']*node['nleft']+node['eright']*node['nright']
    err_node = e_node/(node['nleft']+node['nright'])+alpha
    e_subtree, num_leaves = traverse_subtree(node)
    err_tnode = e_subtree/(node['nleft']+node['nright'])+alpha*num_leaves
    if err_node<err_tnode:
        node['left'] = node['gleft']
        node['right'] = node['gright']
    else:
        if isinstance(node['left'], dict):
            prune_tree(node['left'], alpha)
        if isinstance(node['right'], dict):
            prune_tree(node['right'], alpha)
    return node

# Classification and Regression Tree Algorithm
def decision_tree(train, test, max_depth, min_size, a, b, alpha):
	tree = build_tree(train, max_depth, min_size, a, b)
	tree = prune_tree(tree, alpha)
	err, size = traverse_subtree(tree)
	predictions = list()
	for row in test:
		prediction = predict(tree, row)
		predictions.append(prediction)
	return(predictions, size)

seed(1)
# load and prepare data
filename = 'Cryotherapy.csv'
dataset = load_csv(filename)
# convert string attributes to integers
for i in range(len(dataset[0])):
 	str_column_to_float(dataset, i)
    
# X, y = load_iris(return_X_y=True)
# X, y = load_digits(return_X_y=True)
# dataset = []
# for i in range(len(X)):
#     Z = list(X[i])
#     Z.append(y[i])
#     dataset.append(Z)
# evaluate algorithm
n_folds = 5
max_depth = 5
min_size = 10

mean_sd = []

arange = [0.5,0.65,0.8,0.95,1.1,1.25,1.4,1.55,1.7,1.85,2.0,2.15,2.3,2.45,2.6,2.75]
brange = [1, 2, 3, 4, 5, 6, 7, 8]
alpha_range = [0.005,0.01,0.02,0.025,0.03]

for a in arange:
    print('a='+str(a))
    msd =[]
    for b in brange:
        # print('a='+str(a)+',b='+str(b))
        # root = build_tree(dataset, max_depth, min_size, a, b)
        scores, sizes = evaluate_algorithm(dataset, decision_tree, n_folds, max_depth, min_size, a, b, 0)
        # print('Scores: %s' % scores)
        m = sum(scores)/float(len(scores))
        ms = sum(sizes)/float(len(sizes))
        
        sd = scipy.stats.sem(scores)
        confidence = 0.95
        h = sd * scipy.stats.t.ppf((1 + confidence) / 2., n_folds-1)
        # print('Mean Accuracy: %.2f+-%.2f%%' % (m, h))
        msd.append([m,h,ms])
    mean_sd.append(msd)


        
means = []
for a in range(len(mean_sd)):
    mean = []
    for b in range(len(mean_sd[0])):
        mean.append(mean_sd[a][b][0])
    means.append(mean)
means = []
eta = 0.1
for a in range(len(mean_sd)):
    mean = []
    for b in range(len(mean_sd[0])):
        mean.append(1-mean_sd[a][b][0]/100+eta*mean_sd[a][b][2])
    means.append(mean)
    

# adding labels to x and y axis
plt.xlabel(r"$\beta$",
              size = 8)
plt.ylabel(r"$\alpha$",
              size= 16)
# adding plot title
# plt.title(r"Accuracy for ($\alpha,\beta$)-Tsallis splitting criterion")
plt.title("Cryotherapy")
# adjusting the ticks on both x and y axis
plt.xticks(np.arange(8), brange)
plt.yticks(np.arange(16), arange)


plt.imshow(means, interpolation='none')
plt.show()
        
